# All Copyrights Reserved to Solid Super Solutions

from odoo import models, _
from odoo.exceptions import UserError
import io
import logging

_logger = logging.getLogger(__name__)

try:
    from PyPDF2 import PdfFileReader, PdfFileWriter
except ImportError:
    try:
        from PyPDF2 import PdfReader as PdfFileReader, PdfWriter as PdfFileWriter
    except ImportError:
        PdfFileReader = None
        PdfFileWriter = None


class IrActionsReport(models.Model):
    _inherit = 'ir.actions.report'

    def _render_qweb_pdf_prepare_streams(self, report_ref, data, res_ids=None):
        """
        Override to handle PDF generation for stock card report with multiple products.
        The problem: When using t-foreach on 'data' (list of dictionaries) with web.external_layout,
        Odoo has issues merging multiple PDFs correctly.
        Solution: Generate PDF for the entire report at once, but validate it's correct.
        """
        # Only handle our custom stock card report
        if report_ref == 'sss_stock_card_report.stock_card_report':
            if PdfFileReader is None or PdfFileWriter is None:
                _logger.warning("PyPDF2 not available, using default PDF generation")
                return super()._render_qweb_pdf_prepare_streams(report_ref, data, res_ids=res_ids)
            
            try:
                # Get the wizard to check how many products we have
                if res_ids and len(res_ids) == 1:
                    wizard = self.env['stock.card.wizard'].browse(res_ids[0])
                    if wizard.exists():
                        # Get report data to check number of products
                        report_model = self.env['report.sss_stock_card_report.stock_card_report']
                        report_data = report_model._get_report_data(wizard) or []
                        
                        # If we have multiple products, we need special handling
                        if len(report_data) > 1:
                            # Generate PDF for all products at once
                            # Odoo should handle this, but we'll validate the result
                            result = super()._render_qweb_pdf_prepare_streams(report_ref, data, res_ids=res_ids)
                            
                            # Validate and fix the PDF if needed
                            if result and isinstance(result, dict):
                                for key, value in result.items():
                                    if isinstance(value, dict) and 'stream' in value:
                                        stream = value['stream']
                                        if stream:
                                            try:
                                                # Check if it's a valid PDF
                                                stream.seek(0)
                                                peek = stream.read(4)
                                                stream.seek(0)
                                                
                                                if peek and len(peek) >= 4 and peek[:4] == b'%PDF':
                                                    # Valid PDF header, try to read it
                                                    try:
                                                        pdf_reader = PdfFileReader(stream, strict=False)
                                                        num_pages = pdf_reader.getNumPages()
                                                        stream.seek(0)
                                                        
                                                        if num_pages > 0:
                                                            # Valid PDF with pages, return it
                                                            return result
                                                        else:
                                                            # Empty PDF, try to regenerate
                                                            _logger.warning("Empty PDF generated, regenerating...")
                                                    except Exception as e:
                                                        _logger.warning("PDF validation error: %s, regenerating...", str(e))
                                                else:
                                                    # Invalid PDF header, try to regenerate
                                                    _logger.warning("Invalid PDF header, regenerating...")
                                                    
                                                # If we get here, PDF is invalid, try regeneration
                                                stream.seek(0)
                                                return super()._render_qweb_pdf_prepare_streams(report_ref, data, res_ids=res_ids)
                                            except Exception as e:
                                                _logger.error("Error validating PDF: %s", str(e))
                                                # Try regeneration
                                                return super()._render_qweb_pdf_prepare_streams(report_ref, data, res_ids=res_ids)
                            
                            return result
                
                # For single product or default case, use parent method
                return super()._render_qweb_pdf_prepare_streams(report_ref, data, res_ids=res_ids)
                
            except Exception as e:
                _logger.error("Error in stock card PDF generation: %s", str(e))
                # Fall back to default behavior
                return super()._render_qweb_pdf_prepare_streams(report_ref, data, res_ids=res_ids)
        
        # For all other reports, use default behavior
        return super()._render_qweb_pdf_prepare_streams(report_ref, data, res_ids=res_ids)
