# -*- coding: utf-8 -*-
from odoo import models, api
from odoo.exceptions import ValidationError
from odoo.tools.translate import _
from odoo.tools import float_round


class StockCardReport(models.AbstractModel):
    _name = 'report.sss_stock_card_report.stock_card_report'
    _description = 'Stock Card Report'

    # Maximum products allowed for PDF report to prevent memory issues
    MAX_PRODUCTS_FOR_PDF = 50
    # Maximum moves per product to prevent memory issues
    MAX_MOVES_PER_PRODUCT = 1000

    def _get_stock_moves(self, product, date_from, date_to, warehouse):
        """Get stock moves for a product with optimized query"""
        domain = [
            ('product_id', '=', product.id),
            ('state', '=', 'done'),
            ('date', '>=', date_from),
            ('date', '<=', date_to),
            '|',
                ('location_id.warehouse_id', '=', warehouse.id),
                ('location_dest_id.warehouse_id', '=', warehouse.id)
        ]
        moves = self.env['stock.move'].search(domain, order='date')
        return moves

    def _format_number(self, value):
        """Format number with proper decimal places (always 2 decimal places)"""
        if value is None:
            return 0.0
        return float_round(float(value), precision_digits=2)

    def _get_opening_balance(self, product, start_date, warehouse):
        """Calculate opening balance with optimized query using read_group"""
        domain = [
            ('product_id', '=', product.id),
            ('state', '=', 'done'),
            ('date', '<', start_date),
            '|',
                ('location_id.warehouse_id', '=', warehouse.id),
                ('location_dest_id.warehouse_id', '=', warehouse.id)
        ]
        
        # Use read_group for better performance on large datasets
        try:
            # Get sum of incoming moves
            incoming = self.env['stock.move'].read_group(
                domain + [('location_dest_id.warehouse_id', '=', warehouse.id)],
                ['product_qty:sum'],
                []
            )
            incoming_qty = incoming[0]['product_qty'] if incoming and incoming[0].get('product_qty') else 0.0
            
            # Get sum of outgoing moves
            outgoing = self.env['stock.move'].read_group(
                domain + [('location_id.warehouse_id', '=', warehouse.id)],
                ['product_qty:sum'],
                []
            )
            outgoing_qty = outgoing[0]['product_qty'] if outgoing and outgoing[0].get('product_qty') else 0.0
            
            balance = incoming_qty - outgoing_qty
        except Exception:
            # Fallback to original method if read_group fails
            opening_moves = self.env['stock.move'].search(domain)
            balance = 0.0
            for move in opening_moves:
                if move.location_dest_id.warehouse_id.id == warehouse.id:
                    balance += move.product_qty
                if move.location_id.warehouse_id.id == warehouse.id:
                    balance -= move.product_qty
        
        return self._format_number(balance)

    def _get_stock_status(self, balance, product):
        """
        Determine stock status based on balance and product thresholds
        Returns: tuple(status_code, status_text)
        status_code: 'low', 'normal', or 'high'
        """
        low_threshold = product.stock_card_low_threshold
        normal_threshold = product.stock_card_normal_threshold

        if balance < low_threshold:
            return 'low', _('Low Stock')
        elif balance < normal_threshold:
            return 'normal', _('Normal')
        else:
            return 'high', _('High Stock')

    def _get_report_data(self, wizard):
        """Get report data with memory optimization and limits"""
        if wizard.report_by == 'product':
            if not wizard.product_ids:
                raise ValidationError(_('Please select at least one product.'))
            
            products = wizard.product_ids
            
            # Check limit for product selection
            if len(products) > self.MAX_PRODUCTS_FOR_PDF:
                raise ValidationError(
                    _('Too many products selected (%d). Maximum %d products allowed for PDF reports. '
                      'Please select fewer products or use Excel export for larger datasets.')
                    % (len(products), self.MAX_PRODUCTS_FOR_PDF)
                )
        else:
            if not wizard.category_ids:
                raise ValidationError(_('Please select at least one product category.'))
            
            # Get all category IDs including child categories
            all_category_ids = list(wizard.category_ids.ids)
            for category in wizard.category_ids:
                # Include child categories recursively using child_of operator
                child_categories = self.env['product.category'].search([
                    ('id', 'child_of', category.id)
                ])
                all_category_ids.extend(child_categories.ids)
            # Remove duplicates
            all_category_ids = list(set(all_category_ids))

            # Search for products in selected categories
            # In Odoo 19, products with type='consu' are Goods (storable/consumable)
            products = self.env['product.product'].search([
                '|',
                ('categ_id', 'in', all_category_ids),
                ('product_tmpl_id.categ_id', 'in', all_category_ids),
                ('type', '=', 'consu')
            ])
            
            if not products:
                raise ValidationError(_('No products found in the selected categories.'))
            
            # Check limit for category selection
            if len(products) > self.MAX_PRODUCTS_FOR_PDF:
                raise ValidationError(
                    _('Too many products found in selected categories (%d products). '
                      'Maximum %d products allowed for PDF reports. '
                      'Please select more specific categories or use Excel export for larger datasets.')
                    % (len(products), self.MAX_PRODUCTS_FOR_PDF)
                )

        result = []

        for product in products:
            # Get moves with limit to prevent memory issues
            moves = self._get_stock_moves(product, wizard.start_date, wizard.end_date, wizard.warehouse_id)
            
            # Limit moves per product to prevent memory issues
            moves_count = len(moves)
            if moves_count > self.MAX_MOVES_PER_PRODUCT:
                # Limit moves but continue processing
                moves = moves[:self.MAX_MOVES_PER_PRODUCT]
            
            balance = self._get_opening_balance(product, wizard.start_date, wizard.warehouse_id)
            move_lines = []
            
            # Add opening balance line
            move_lines.append({
                'date': wizard.start_date,
                'origin': 'Opening Balance',
                'picking_number': False,
                'quantity_in': self._format_number(0),
                'quantity_out': self._format_number(0),
                'balance': self._format_number(balance),
            })
            
            # Process moves efficiently
            for move in moves:
                qty_in = qty_out = 0
                
                if move.location_dest_id.warehouse_id.id == wizard.warehouse_id.id:
                    qty_in = move.product_qty
                    balance += move.product_qty
                
                if move.location_id.warehouse_id.id == wizard.warehouse_id.id:
                    qty_out = move.product_qty
                    balance -= move.product_qty
                
                # Get picking number and origin
                picking_id = move.picking_id
                picking_number = picking_id.name if picking_id else ''
                origin = picking_id.origin if picking_id else ''
                
                display_name = picking_number
                if origin and origin != picking_number:
                    display_name = f"{picking_number} ({origin})" if picking_number else origin
                
                move_lines.append({
                    'date': move.date,
                    'origin': display_name or move.origin or move.reference or '',
                    'picking_number': picking_number,
                    'quantity_in': self._format_number(qty_in),
                    'quantity_out': self._format_number(qty_out),
                    'balance': self._format_number(balance),
                })

            current_balance = move_lines[-1]['balance'] if move_lines else 0.0
            status_code, status_text = self._get_stock_status(current_balance, product)

            result.append({
                'product': product,
                'moves': move_lines,
                'stock_status': {
                    'code': status_code,
                    'text': status_text,
                    'balance': current_balance
                }
            })

        return result

    @api.model
    def _get_report_values(self, docids, data=None):
        if not docids:
            raise ValidationError(_('No report data found.'))

        wizard = self.env['stock.card.wizard'].browse(docids)

        if not wizard:
            raise ValidationError(_('Report wizard not found.'))

        return {
            'doc_ids': docids,
            'doc_model': 'stock.card.wizard',
            'docs': wizard,
            'data': self._get_report_data(wizard),
        }
