# -*- coding: utf-8 -*-
from odoo import models, fields, api, _
from odoo.exceptions import UserError


class AccountAsset(models.Model):
    _inherit = 'account.asset'
    _mail_post_access = 'read'

    state = fields.Selection(selection_add=[('cancelled', 'Cancelled')])
    cancellation_reason = fields.Text(string='Cancellation Reason', readonly=True, track_visibility='onchange')
    cancelled_by = fields.Many2one('res.users', string='Cancelled By', readonly=True, track_visibility='onchange')
    cancellation_date = fields.Date(string='Cancellation Date', readonly=True, track_visibility='onchange')

    def action_cancel_asset(self, reason=False):
        """Cancel the asset and reverse all related journal entries."""
        self.ensure_one()
        if self.state == 'cancelled':
            raise UserError(_('Asset is already cancelled.'))
        
        if self.state not in ['open', 'paused']:
            raise UserError(_('Only running or paused assets can be cancelled.'))

        if not reason:
            raise UserError(_('Please provide a reason for cancellation.'))

        # Find all related journal entries
        moves = self.depreciation_move_ids.filtered(lambda m: m.state == 'posted')
        
        if not moves:
            self._set_cancelled_state(reason)
            return True

        # Create reverse entries for each move
        for move in moves:
            # In Odoo 18, create reversal manually by reversing debit/credit
            reverse_line_vals = []
            for line in move.line_ids:
                line_vals = {
                    'name': line.name,
                    'account_id': line.account_id.id,
                    'debit': line.credit,
                    'credit': line.debit,
                }
                # Add partner if exists
                if line.partner_id:
                    line_vals['partner_id'] = line.partner_id.id
                # Add analytic distribution if exists
                if line.analytic_distribution:
                    line_vals['analytic_distribution'] = line.analytic_distribution
                reverse_line_vals.append((0, 0, line_vals))
            
            reverse_move = self.env['account.move'].create({
                'move_type': 'entry',
                'journal_id': move.journal_id.id,
                'date': move.date,
                'ref': _('Reversal of: %s (Wrong Asset - %s)') % (move.name, reason),
                'asset_id': self.id,  # Link reversal move to asset so it appears in depreciation board
                'line_ids': reverse_line_vals,
            })
            
            # Post the reversal entry
            if reverse_move and reverse_move.state == 'draft':
                reverse_move.action_post()

        # Delete all unposted (draft) moves
        unposted_moves = self.depreciation_move_ids.filtered(lambda m: m.state == 'draft')
        if unposted_moves:
            unposted_moves.unlink()

        # Update to cancelled state with reason
        self._set_cancelled_state(reason)
        
        # Log the cancellation in the chatter
        self.message_post(
            body=_('Asset cancelled by %s with reason: %s') % (self.env.user.name, reason),
            subtype_id=self.env.ref('mail.mt_note').id
        )
        
        return True

    def _set_cancelled_state(self, reason):
        """Helper method to set cancelled state with tracking information."""
        self.write({
            'state': 'cancelled',
            'cancellation_reason': reason,
            'cancelled_by': self.env.user.id,
            'cancellation_date': fields.Date.today(),
        })

    def toggle_active(self):
        """Override to allow archiving cancelled assets."""
        for record in self:
            if record.state not in ['close', 'cancelled']:
                raise UserError(_('You cannot archive a record that is not closed or cancelled'))
        return super(AccountAsset, self).toggle_active() 