from odoo import models, api

class AccountMove(models.Model):
    _inherit = 'account.move'

    def print_journal_report(self):
        self.ensure_one()
        return self.env.ref('sss_simple_journal_report.action_simple_journal_report').report_action(self)
    
    def get_report_state_name(self):
        """Get the state name for report file naming.
        
        Returns the state name in English (Draft, Posted, Cancel) for better readability.
        """
        self.ensure_one()
        state_mapping = {
            'draft': 'Draft',
            'posted': 'Posted',
            'cancel': 'Cancel',
        }
        return state_mapping.get(self.state, self.state)
    
    def get_report_file_name(self):
        """Get the report file name based on language and state.
        
        Returns the file name with proper state name. Replaces False/empty name with Draft.
        """
        self.ensure_one()
        # Handle False or empty name for draft entries
        entry_name = self.name
        if not entry_name or entry_name in (False, '/'):
            if self.env.lang == 'ar_001':
                entry_name = 'مسودة'
            else:
                entry_name = 'Draft'
        
        if self.env.lang == 'ar_001':
            if self.state == 'draft':
                return f'قيد اليومية - {entry_name}'
            else:
                return f'قيد اليومية - {entry_name} - {self.get_report_state_name()}'
        else:
            if self.state == 'draft':
                return f'Journal Entry - {entry_name}'
            else:
                return f'Journal Entry - {entry_name} - {self.get_report_state_name()}'

class AccountMoveLine(models.Model):
    _inherit = 'account.move.line'

    def get_formatted_analytic_distribution(self):
        """Format analytic distribution for display in reports.
        
        Returns a formatted string with account names and percentages.
        Compatible with Odoo 19 analytic distribution structure.
        """
        self.ensure_one()
        if not self.analytic_distribution:
            return ''
        
        try:
            formatted_dist = []
            # Handle both string and integer keys (Odoo 19 compatibility)
            account_ids = []
            for key in self.analytic_distribution.keys():
                try:
                    account_ids.append(int(key))
                except (ValueError, TypeError):
                    continue
            
            if not account_ids:
                return ''
            
            analytic_accounts = self.env['account.analytic.account'].browse(account_ids)
            # Create mapping for both string and integer keys
            account_dict = {}
            for acc in analytic_accounts:
                account_dict[str(acc.id)] = acc.name
                account_dict[acc.id] = acc.name
            
            for account_id, percentage in self.analytic_distribution.items():
                # Handle both string and integer keys
                key = account_id if account_id in account_dict else str(account_id)
                name = account_dict.get(key, '')
                if name:
                    formatted_dist.append(f"{name}: {percentage}%")
                    
            return "\n".join(formatted_dist) if formatted_dist else ''
        except Exception:
            # Fallback: return empty string if any error occurs
            return '' 